#!/usr/bin/lua

-- Created by jiaqiang.ye@lnxall.com
-- 2023/12/14
-- Firmware installation for Ubuntu-18.04

local bit32 = require "bit32"
local posix = require "posix"
local invoker = require "invoker"

local cfmt = string.format

local G_BLKDEV = nil

local function logmsg(msg)
	local kfd = posix.open("/dev/kmsg", bit32.bor(posix.O_WRONLY, posix.O_CLOEXEC))
	if type(kfd) ~= "number" or kfd < 0 then
		io.stderr:write("Error, failed to open kmsg.\n")
		io.stderr:flush()
		return false
	end

	local mlen = string.len(msg)
	if mlen ~= posix.write(kfd, msg) then
		io.stdout:write("Error, kmsg write has failed.\n")
		io.stdout:write(msg); io.stdout:flush()
	end
	posix.close(kfd); kfd = -1
	return true
end

local diskpart = function (pno)
	local i, j = 1, 1
	while i < 16 do
		local part = cfmt("%s%d", G_BLKDEV, i)
		if posix.access(part) == 0 then
			if j >= pno then return part, i end
			j = j + 1
		end
		i = i + 1
	end
	return '/dev/not-found', 0
end

local diskpart_nvme = function (pno)
	local i, j = 1, 1
	while i < 16 do
		local part = cfmt("%sp%d", G_BLKDEV, i)
		if posix.access(part) == 0 then
			if j >= pno then return part, i end
			j = j + 1
		end
		i = i + 1
	end
	return '/dev/not-found', 0
end


local function detect_harddisk(prefix)
	local okay, dfout = invoker.invoke(invoker.OUTPUT, "df", "-h")
	if okay ~= 0 or type(dfout) ~= "string" then
		logmsg("Error, `df -h has no output.\n")
		return false
	end

	local hiter = { "a", "b", "c", "d", "e", "f" }
	for _, hi in ipairs(hiter) do
		local disk = prefix .. hi
		local dst = posix.stat(disk)
		if type(dst) == "table" and not string.find(dfout, disk, 1, true) then
			G_BLKDEV = disk
			logmsg(cfmt("Will Install to harddisk: %s\n", disk))
			return true
		end
	end

	local disk = '/dev/nvme0n1'
	local dst = posix.stat(disk)
	if type(dst) == "table" and not string.find(dfout, disk, 1, true) then
		G_BLKDEV = disk
		logmsg(cfmt("Will Install to harddisk: %s\n", disk))
		diskpart = diskpart_nvme
		return true
	end
	return false
end

local function write_ub1804(img)
	local ist = posix.stat(img)
	if type(ist) ~= "table" then
		logmsg(cfmt("Error, Ubuntu-18.04 disk image not found: %s\n", img))
		return false
	end

	io.stdout:write(cfmt("Unpacking Ubuntu-18.04 image '%s' to '%s'\n", img, G_BLKDEV))
	io.stdout:flush()
	local okay, output = invoker.invoke(invoker.OUTPUT, "/bin/sh", "-c",
		cfmt("xzcat -T 4 '%s' | dd of=%s bs=262144 conv=fsync,notrunc", img, G_BLKDEV))
	if not okay or okay ~= 0 then
		if type(okay) ~= "number" then okay = -1 end
		if type(output) ~= "string" then output = "unknown error" end
		logmsg(cfmt("Error, failed to unpack Ubuntu-18.04 image: %d => %s\n", okay, output))
		return false
	end
	posix.sync()
	logmsg("\nUbuntu-18.04 Image written successfully!\n")
	return true
end

local function expand_rootfs()
	logmsg(cfmt("\nRefreshing kernel awareness of partition '%s'...\n", G_BLKDEV))
	local okay = invoker.invoke(0, "partx", G_BLKDEV)
	if okay ~= 0 then
		logmsg("\nWarning, Refreshing has failed.\n")
	end

	local blk2, idx = diskpart(2)
	logmsg(cfmt("\nExpanding partition '%s'...\n", blk2))
	okay = invoker.invoke(0, "parted", "-s", "-a", "opt", G_BLKDEV, cfmt("resizepart %d 100%%", idx))
	if okay ~= 0 then
		logmsg(cfmt("Error, failed to expand partition '%s'.\n", blk2))
		return false
	end

	invoker.invoke(invoker.NOSTDIO, "partx", G_BLKDEV)
	invoker.waitsec(2)

	local rootp, pdx = diskpart(4)
	okay = invoker.invoke(0, "parted", "-s", "-a", "opt", G_BLKDEV, cfmt("resizepart %d 100%%", pdx))
	if okay ~= 0 then
		logmsg(cfmt("Error, failed to expand extended partition '%s'.\n", rootp))
	end

	logmsg(cfmt("\nChecking Ubuntu-18.04 root/ext4 file system: %s\n", rootp))
	invoker.invoke(invoker.OUTPUT + invoker.NOSTDIO, "e2fsck", "-p", "-f", rootp)

	logmsg("\nExpanding ext4 file system for Ubuntu-18.04...\n")
	okay = invoker.invoke(0, "resize2fs", rootp)
	if okay ~= 0 then
		logmsg("Error, failed to expand ext4!\n")
		return false
	end

	posix.sync()
	logmsg("Done!\n")
	return true
end

local function mainfunc()
	posix.chdir('/')
	posix.signal(posix.SIGPIPE, posix.SIG_IGN)
	local nowt = invoker.uptime()
	if nowt < 45 then invoker.waitsec(45 - nowt) end
	invoker.invoke(invoker.NOSTDIO, "/bin/sh", "-c", "echo '8 3 1 7' > /proc/sys/kernel/printk")
	invoker.invoke(invoker.NOSTDIO, "/bin/sh", "-c", "echo 'on' > /proc/sys/kernel/printk_devkmsg")
	if not detect_harddisk("/dev/sd") then
		return false
	end

	if not write_ub1804("/root/ub1804-ems2.0.img.xz") then
		return false
	end

	return expand_rootfs()
end

if mainfunc() then
	logmsg("        :'#######::'##:::'##::::'###::::'##:::'##:\n")
	logmsg("        '##.... ##: ##::'##::::'## ##:::. ##:'##::\n")
	logmsg("         ##:::: ##: ##:'##::::'##:. ##:::. ####:::\n")
	logmsg("         ##:::: ##: #####::::'##:::. ##:::. ##::::\n")
	logmsg("         ##:::: ##: ##. ##::: #########:::: ##::::\n")
	logmsg("         ##:::: ##: ##:. ##:: ##.... ##:::: ##::::\n")
	logmsg("        . #######:: ##::. ##: ##:::: ##:::: ##::::\n")
	logmsg("        :.......:::..::::..::..:::::..:::::..:::::\n")
else
	logmsg("        '########:'########::'########:::'#######::'########::\n")
	logmsg("         ##.....:: ##.... ##: ##.... ##:'##.... ##: ##.... ##:\n")
	logmsg("         ##::::::: ##:::: ##: ##:::: ##: ##:::: ##: ##:::: ##:\n")
	logmsg("         ######::: ########:: ########:: ##:::: ##: ########::\n")
	logmsg("         ##...:::: ##.. ##::: ##.. ##::: ##:::: ##: ##.. ##:::\n")
	logmsg("         ##::::::: ##::. ##:: ##::. ##:: ##:::: ##: ##::. ##::\n")
	logmsg("         ########: ##:::. ##: ##:::. ##:. #######:: ##:::. ##:\n")
	logmsg("        ........::..:::::..::..:::::..:::.......:::..:::::..::\n")
end

posix.sync()
while true do invoker.waitsec(60) end
