#!/usr/bin/lua

-- Created by jiaqiang.ye@lnxall.com
-- 2023/12/14
-- Firmware installation for CentOS7

local bit32 = require "bit32"
local posix = require "posix"
local invoker = require "invoker"

local cfmt = string.format

local G_BLKDEV = nil

local function logmsg(msg)
	local kfd = posix.open("/dev/kmsg", bit32.bor(posix.O_WRONLY, posix.O_CLOEXEC))
	if type(kfd) ~= "number" or kfd < 0 then
		io.stderr:write("Error, failed to open kmsg.\n")
		io.stderr:flush()
		return false
	end

	local mlen = string.len(msg)
	if mlen ~= posix.write(kfd, msg) then
		io.stdout:write("Error, kmsg write has failed.\n")
		io.stdout:write(msg); io.stdout:flush()
	end
	posix.close(kfd); kfd = -1
	return true
end

local function detect_harddisk(prefix)
	local okay, dfout = invoker.invoke(invoker.OUTPUT, "df", "-h")
	if okay ~= 0 or type(dfout) ~= "string" then
		logmsg("Error, `df -h has no output.\n")
		return false
	end

	local hiter = { "a", "b", "c", "d", "e", "f" }
	for _, hi in ipairs(hiter) do
		local disk = prefix .. hi
		local dst = posix.stat(disk)
		if type(dst) == "table" and not string.find(dfout, disk, 1, true) then
			G_BLKDEV = disk
			logmsg(cfmt("Will Install to harddisk: %s\n", disk))
			return true
		end
	end
	return false
end

local function write_centos7(img)
	local ist = posix.stat(img)
	if type(ist) ~= "table" then
		logmsg(cfmt("Error, CentOS-7 disk image not found: %s\n", img))
		return false
	end

	io.stdout:write(cfmt("Unpacking CentOS-7 image '%s' to '%s'\n", img, G_BLKDEV))
	io.stdout:flush()
	local okay, output = invoker.invoke(invoker.OUTPUT, "/bin/sh", "-c",
		cfmt("xzcat -T 4 '%s' | dd of=%s bs=256k conv=fsync,notrunc", img, G_BLKDEV))
	if not okay or okay ~= 0 then
		if type(okay) ~= "number" then okay = -1 end
		if type(output) ~= "string" then output = "unknown error" end
		logmsg(cfmt("Error, failed to unpack CentOS-7 image: %d => %s\n", okay, output))
		return false
	end
	posix.sync()
	logmsg("\nCentOS-7 Image written successfully!\n")
	return true
end

local function expand_rootfs()
	logmsg(cfmt("\nRefreshing kernel awareness of partition '%s'...\n", G_BLKDEV))
	local okay = invoker.invoke(0, "partx", G_BLKDEV)
	if okay ~= 0 then
		logmsg("\nWarning, Refreshing has failed.\n")
	end

	logmsg(cfmt("\nDeleting partition '%s3'...\n", G_BLKDEV))
	invoker.invoke(invoker.NOSTDIO, "parted", G_BLKDEV, "rm", "3")
	invoker.invoke(invoker.NOSTDIO, "parted", G_BLKDEV, "rm", "3")

	local blk2 = G_BLKDEV .. "2"
	logmsg(cfmt("\nExpanding partition '%s'...\n", blk2))
	okay = invoker.invoke(0, "parted", "-s", "-a", "opt", G_BLKDEV, "resizepart 2 100%")
	if okay ~= 0 then
		logmsg(cfmt("Error, failed to expand partition '%s'.\n", blk2))
		return false
	end

	logmsg(cfmt("\nExpand logical Volume '%s'...\n", blk2))
	okay = invoker.invoke(0, "pvresize", blk2)
	if okay ~= 0 then
		logmsg("Error, failed to expand logical Volume.\n")
		return false
	end

	local rootp = "/dev/mapper/centos-root"
	logmsg("\nExpanding centos-root sub-partition...\n")
	okay = invoker.invoke(0, "lvextend", "-l", "+100%FREE", rootp)
	if okay ~= 0 then
		logmsg("Error, failed to expand centos-root partition.\n")
		return false
	end

	logmsg("\nMounting centos-root as XFS file system...\n")
	local mdir = "/tmp/mount_dir"
	invoker.invoke(invoker.NOSTDIO, "mkdir", "-p", mdir)
	invoker.invoke(invoker.NOSTDIO, "umount", mdir)
	okay = invoker.invoke(0, "mount", "-orw", "-t", "xfs", rootp, mdir)
	if okay ~= 0 then
		logmsg(cfmt("Error, failed to mount %s as XFS.\n", blk2))
		return false
	end

	logmsg("\nExpanding XFS file system for CentOS-7...\n")
	okay = invoker.invoke(0, "xfs_growfs", mdir)
	if okay ~= 0 then
		invoker.invoke(invoker.NOSTDIO, "umount", mdir)
		logmsg("Error, failed to expand XFS!\n")
		return false
	end

	logmsg("Done!\n")
	invoker.invoke(invoker.NOSTDIO, "umount", mdir)
	return true
end

local function mainfunc()
	local nowt = invoker.uptime()
	if nowt < 75 then invoker.waitsec(75 - nowt) end
	invoker.invoke(invoker.NOSTDIO, "/bin/sh", "-c", "echo '8 3 1 7' > /proc/sys/kernel/printk")
	invoker.invoke(invoker.NOSTDIO, "/bin/sh", "-c", "echo 'on' > /proc/sys/kernel/printk_devkmsg")
	if not detect_harddisk("/dev/sd") then
		return false
	end

	if not write_centos7("/root/CentOS7-dd-image.xz") then
		return false
	end

	return expand_rootfs()
end

if mainfunc() then
	logmsg("        :'#######::'##:::'##::::'###::::'##:::'##:\n")
	logmsg("        '##.... ##: ##::'##::::'## ##:::. ##:'##::\n")
	logmsg("         ##:::: ##: ##:'##::::'##:. ##:::. ####:::\n")
	logmsg("         ##:::: ##: #####::::'##:::. ##:::. ##::::\n")
	logmsg("         ##:::: ##: ##. ##::: #########:::: ##::::\n")
	logmsg("         ##:::: ##: ##:. ##:: ##.... ##:::: ##::::\n")
	logmsg("        . #######:: ##::. ##: ##:::: ##:::: ##::::\n")
	logmsg("        :.......:::..::::..::..:::::..:::::..:::::\n")
else
	logmsg("        '########:'########::'########:::'#######::'########::\n")
	logmsg("         ##.....:: ##.... ##: ##.... ##:'##.... ##: ##.... ##:\n")
	logmsg("         ##::::::: ##:::: ##: ##:::: ##: ##:::: ##: ##:::: ##:\n")
	logmsg("         ######::: ########:: ########:: ##:::: ##: ########::\n")
	logmsg("         ##...:::: ##.. ##::: ##.. ##::: ##:::: ##: ##.. ##:::\n")
	logmsg("         ##::::::: ##::. ##:: ##::. ##:: ##:::: ##: ##::. ##::\n")
	logmsg("         ########: ##:::. ##: ##:::. ##:. #######:: ##:::. ##:\n")
	logmsg("        ........::..:::::..::..:::::..:::.......:::..:::::..::\n")
end

posix.sync()
while true do invoker.waitsec(60) end
