#!/opt/lnxall/bin/lua

-- Created by jiaqiang.ye@lnxall.com
-- 4G network for CentOS-7
-- The linux kernel version must be:
--   uname -r
--   3.10.0-1160.88.1.el7.x86_64
-- 2023/03/13

-- Load external modules
local bit32 = require 'bit32'
local posix = require 'posix'
local cjson = require 'cjson'
local invoker = require 'invoker'

-- global CONSTANT variables
local gfmt = string.format
local LAN_NETDEV = nil -- if LAN_NETDEV is not nil, try to power off/on when 4G is abnormal
local TTY_MODEM = "/tmp/4g_modem_tty"
local LOG_FILE = "/var/log/lnxall.log"
local CHECK_INTERVAL = 180 -- check-interval for 4G network connection
local PING_ADDRS = { [1] = 'emsmqtt.lnxall.com', [2] = 'smc.lnxall.com',
	[3] = 'emsiot.lnxall.com', [4] = 'mqtt.lnxall.com', [5] = 'www.google.com' }

-- global varibales
local g_usbid = nil   -- USB ID of 4G network modem
local g_netdev = nil  -- name of USB network device
local g_4ginfo = {}   -- Array of supported USB modem
local g_metrics = { [1] = 50, [2] = 150 }
local g_dns = { [1] = '8.8.8.8', [2] = '114.114.114.114' }

g_4ginfo['2c7c:6005'] = { ['ttyindex'] = 2, ['script'] = 'ec200.gcom',  ['dhcp'] = true }
g_4ginfo['2c7c:6026'] = { ['ttyindex'] = 2, ['script'] = 'ec200.gcom',  ['dhcp'] = true }
g_4ginfo['2c7c:0125'] = { ['ttyindex'] = 2, ['script'] = 'ec2x.gcom',  ['dhcp'] = true }
g_4ginfo['1782:4e00'] = { ['ttyindex'] = 1, ['script'] = 'air720.gcom', ['dhcp'] = true }

local function netdev_exists(iface)
	local rval = invoker.invoke(invoker.NOSTDIO,
		"ip", "link", "show", "dev", iface)
	return rval == 0
end

local function load_json(jp)
	local pfil = io.open(jp, "rb")
	if not pfil then return nil end
	local pdat = pfil:read("*a")
	pfil:close(); pfil = nil
	if type(pdat) ~= "string" or #pdat == 0 then return nil end
	local okay, jdat = pcall(cjson.decode, pdat)
	if not okay then return nil end
	return jdat
end

local function write_json(jd, jp)
	if type(jd) == "table" then jd = cjson.encode(jd) end
	if type(jd) ~= "string" or #jd == 0 then return false end
	local pfil = io.open(jp, "wb")
	if pfil then
		pfil:write(jd); pfil:close(); pfil = nil
		return true
	end
	return false
end

local function find_4g_netdev()
	local okay, output = invoker.invoke(invoker.OUTPUT, "ip", "link", "show")
	if okay ~= 0 or type(output) ~= "string" then
		return nil
	end
	for mdev in string.gmatch(output, "enp[^:\r\n]+:") do
		local nlen = string.len(mdev)
		if nlen >= 9 then return mdev:sub(1, nlen - 1) end
	end
	return nil
end

local function update_sim_iccid(usbdev)
	local outfile = '/run/sim_iccid.txt'
	if posix.access(outfile) == 0 then return true end
	if not usbdev then
		if not g_usbid then return false end
		local i4g = g_4ginfo[g_usbid]
		usbdev = gfmt("/dev/ttyUSB%d", i4g.ttyindex)
	end
	if posix.access(usbdev) ~= 0 then return false end

	invoker.invoke(invoker.NOSTDIO, "stty", "-F", usbdev, "raw", "cs8")
	local usbfd = posix.open(usbdev, posix.O_RDWR + posix.O_NONBLOCK)
	if type(usbfd) ~= "number" or usbfd < 0 then return false end
	posix.write(usbfd, "AT+ICCID\r\n"); invoker.msleep(300)
	posix.write(usbfd, "AT+QICCID\r\n"); invoker.msleep(300)
	local output = posix.read(usbfd, 1024); posix.close(usbfd)
	if type(output) ~= "string" then return false end
	local iccid = string.match(output, "ICCID:%s+(%d+)")
	if iccid then
		local filp = io.open(outfile, "wb")
		if filp then
			filp:write(gfmt("ICCID: %s\n", iccid))
			filp:close(); filp = nil
		end
	end
	return iccid
end

local function write_logfile(logpath, logmsg, maxsize)
	if type(maxsize) ~= "number" then maxsize = 524288 end
	local logst = posix.stat(logpath)
	if type(logst) ~= "table" or logst["type"] ~= "regular" or
		logst["size"] > maxsize then
		-- remove file unconditionally
		invoker.invoke(invoker.NOSTDIO, "rm", "-rf", logpath)
		posix.sync()
	end

	local logf = io.open(logpath, "a")
	if not logf then
		io.stderr:write(gfmt("Error, failed to open file '%s' for appending.\n", logpath))
		io.stderr:flush()
		return false
	end

	logf:write(logmsg)
	logf:close(); logf = nil
	posix.sync()
	return true
end

local function fetch_ttyusb(idx)
	local jdx, kdx = 0, 0
	while jdx < 32 do
		local ndev = gfmt("/dev/ttyUSB%d", jdx)
		local info = posix.stat(ndev)
		if type(info) == "table" and info["type"] == "character device" then
			if kdx >= idx then return ndev end
			kdx = kdx + 1
		end
		jdx = jdx + 1
	end
	return nil
end

local function check_resolv()
	local content = gfmt("# Generated by network_4g.lua\nnameserver %s\nnameserver %s\noptions timeout:2 attempts:2 rotate\n",
		g_dns[1], g_dns[2])
	local rfile = "/etc/resolv.conf"
	local resolv, cont = io.open(rfile, "rb"), nil
	if resolv then
		cont = resolv:read('*a')
		resolv:close(); resolv = nil
	end
	if cont == content then return true end
	-- if cont then io.stdout:write("Warning: old DNS settings:\n" .. cont) end
	write_logfile(LOG_FILE, gfmt("%s: Updating resolv.conf...\n", os.date()))
	resolv = io.open(rfile, "wb")
	if resolv then
		resolv:write(content)
		resolv:close(); resolv = nil
		posix.sync()
		return true
	end
	return false
end

local function pci_usb_bind(path, isbind)
	local regex = "/([^/]+)$"
	local pname = string.match(path, regex)
	pname = gfmt("/run/usb-list-%s.json", pname)
	local ulist = load_json(pname)

	if not ulist then
		local glist = posix.glob(path .. '/*')
		if type(glist) ~= "table" then return false end

		ulist = {}
		for _, filp in ipairs(glist) do
			local filst = posix.stat(filp)
			if type(filst) == "table" and filst['type'] == "link" then
				local pid = string.match(filp, regex)
				if pid then ulist[#ulist + 1] = pid end
			end
		end

		if #ulist == 0 then return false end
		write_json(ulist, pname)
	end

	local bind = gfmt("%s/%s", path, isbind and 'bind' or 'unbind')
	for _, pid in ipairs(ulist) do
		local filp = io.open(bind, "wb")
		if filp then
			io.stdout:write(gfmt("Writing '%s' to '%s'...\n", pid, bind))
			io.stdout:flush()
			filp:write(pid); filp:flush(); filp:close(); filp = nil
		end
	end
	return true
end

local function replace_grub(gp)
	local output = invoker.readfile(gp)
	if type(output) == "string" and (string.find(output, "nomodeset rhgb", 1, true) or
		string.find(output, "swap rhgb")) then
		io.stdout:write("Updating " .. gp .. "\n")
		io.stdout:flush()
		invoker.invoke(0, "sed",
			"-e", "s/nomodeset rhgb/nomodeset rd.driver.blacklist=rndis_host,r8169 rhgb/g",
			"-e", "s/swap rhgb/swap rd.driver.blacklist=rndis_host,r8169 rhgb/g",
			"-i", gp)
	end
end

local function pci_usb_rebind(isbind)
	pci_usb_bind('/sys/bus/pci/drivers/xhci_hcd', isbind)
	pci_usb_bind('/sys/bus/pci/drivers/ehci_hcd', isbind)
	return true
end

local function power_offon_4g(check)
	-- if ttyUSBx is responsive, just return nil
	local udev = g_usbid and g_4ginfo[g_usbid] or nil
	if udev then udev = gfmt('/dev/ttyUSB%d', udev.ttyindex) end
	if check and type(udev) == "string" then
		local okay, output = invoker.invoke(invoker.NOSTDIO + invoker.OUTPUT,
			"comgt", "-d", udev, "-s", "/opt/lnxall/etc/gcom/getstrength.gcom")
		if okay == 0 and type(output) == "string" and string.find(output, "CSQ:", 1, true) then
			write_logfile(LOG_FILE, gfmt("%s: 4G modem is responsive: skip power OFF/ON 4G.\n",
				os.date()))
			return nil
		end
		if type(okay) ~= "number" then okay = 0 end
		write_logfile(LOG_FILE, gfmt("%s: 4G modem is not responsive: %d\n", os.date(), okay))
	end

	local output = invoker.readfile("/proc/net/arp")
	if type(output) ~= "string" or string.len(output) == 0 then return false end

	local ret, regex = false, gfmt("^([%%d.]+)%%s+.+%s", LAN_NETDEV)
	for line in string.gmatch(output, "[^\r\n]+") do
		local ipaddr = string.match(line, regex)
		if type(ipaddr) == "string" and invoker.tcpcheck(ipaddr, 22001, 1000) then
			io.stdout:write(gfmt("Peer TCP connection available at %s:22001\n", ipaddr))
			io.stdout:flush()
			local eval, outmsg = invoker.invoke(invoker.OUTPUT, "sshpass", "-p", "lnxall123", "ssh",
				"-oStrictHostKeyChecking=no", "-oUserKnownHostsFile=/dev/null",
				"-p", "22001", "root@" .. ipaddr,
				"cd /sys/class/gpio/gpio120 && echo 1 > value && sleep 10 && echo 0 > value && echo OKAY ; exit 0")
			if eval == 0 and type(outmsg) == "string" and string.find(outmsg, 'OKAY', 1, true) then
				ret = true
			end
		end
	end
	write_logfile(LOG_FILE, gfmt("%s: power OFF/ON 4G via IMX6UL: %s\n", os.date(), tostring(ret)))
	return ret
end

-- initialize working environment
local function n4g_init(ndev, landev, dns1, dns2)
	-- setup PATH environment variable
	posix.setenv('PATH',
		'/opt/lnxall/bin:/usr/local/bin:/usr/bin:/usr/sbin:/bin:/sbin', true)
	posix.chdir("/")
	invoker.setname('NETWORK_4G')

	if type(dns1) == "string" and #dns1 > 0 then g_dns[1] = dns1 end
	if type(dns2) == "string" and #dns2 > 0 then g_dns[2] = dns2 end

	-- backup old drivers for CentOS-7, 3.10.0-1160.88.1.el7.x86_64 
	local dirm = '/lib/modules/3.10.0-1160.88.1.el7.x86_64/kernel/drivers/usb/serial/'
	if posix.access(dirm .. 'usb_wwan.ko.xz') == 0 then
		posix.rename(dirm .. 'option.ko.xz', dirm .. '.old.option.ko.xz')
		posix.rename(dirm .. 'usb_wwan.ko.xz', dirm .. '.old.usb_wwan.ko.xz')
		posix.sync()
	end
	dirm = '/lib/modules/3.10.0-1160.88.1.el7.x86_64/kernel/drivers/net/usb/'
	if posix.access(dirm .. 'rndis_host.ko.xz') == 0 then
		posix.unlink(dirm .. 'rndis_host.ko.xz')
		posix.sync()
	end

	local insmod = true -- install our kernel modules
	-- check whether our modules have been installed or not
	local okay, outbuf = invoker.invoke(invoker.OUTPUT, "lsmod")
	if okay == 0 and type(outbuf) == "string" and
		string.find(outbuf, 'usb_wwan', 1, true) and
		string.find(outbuf, 'rndis_host', 1, true) and
		string.find(outbuf, 'option', 1, true) then
		insmod = false
	end

	if insmod then
		dirm = '/opt/lnxall/CentOS7-4G/'
		invoker.invoke(invoker.NOSTDIO, "modprobe", "usbnet")
		invoker.invoke(invoker.NOSTDIO, "modprobe", "cdc_ether")
		invoker.invoke(invoker.NOSTDIO, "insmod", dirm .. 'rndis_host.ko')
		invoker.invoke(invoker.NOSTDIO, "insmod", dirm .. 'usb_wwan.ko')
		invoker.invoke(invoker.NOSTDIO, "insmod", dirm .. 'option.ko')
		invoker.waitsec(2)
	end

	-- check for network device for LAN
	if type(landev) == "string" and string.len(landev) > 0 and netdev_exists(landev) then
		LAN_NETDEV = landev
	end

	-- Enumerate output from `lsusb,
	-- and determine the 4G modem USB ID
	okay, outbuf = invoker.invoke(invoker.OUTPUT, "lsusb")
	if okay ~= 0 or type(outbuf) ~= "string" or string.len(outbuf) == 0 then
		io.stderr:write("Error, failed to invoke `lsusb!\n")
		io.stderr:flush()
		return false
	end

	-- process line by line
	for output in string.gmatch(outbuf, "([^\r\n]+)") do
		local usbid = string.match(output, "(%x%x%x%x:%x%x%x%x)")
		if usbid and g_4ginfo[usbid] then
			g_usbid = usbid
			break
		end
	end

	if not g_usbid then
		io.stderr:write("Warning, no 4G network modem found!\n")
		io.stderr:flush()
		return false
	end

	-- write /sys/bus/usb-serial/drivers/option1/new_id
	local idfile = "/sys/bus/usb-serial/drivers/option1/new_id"
	local oldid = invoker.readfile(idfile, invoker.TRIMEND)
	local newid = string.gsub(g_usbid, ":", " ") .. " ff"
	if newid ~= oldid then
		local idh = io.open(idfile, "wb")
		if idh then
			idh:write(newid .. "\n")
			idh:close(); idh = nil
			io.stdout:write(gfmt("Updated option1/new_id with '%s'\n", newid))
			io.stdout:flush()
		end
	end
	idfile, oldid, newid = nil, nil, nil -- clear unused local variables

	local nowtim = invoker.uptime()
	while nowtim < 100 do -- do not check 4G network in such a haste
		check_resolv() -- update resolv.conf before wait
		invoker.waitsec(100 - nowtim)
		nowtim = invoker.uptime()
	end
	check_resolv()

	-- check network device
	if type(ndev) ~= "string" or not netdev_exists(ndev) then
		local newdev = find_4g_netdev()
		if newdev and netdev_exists(newdev) then
			ndev = newdev
			io.stdout:write(gfmt("Using 4G network device: %s\n", ndev))
			io.stdout:flush()
		else
			io.stderr:write(gfmt("Error, network device not found: %s\n",
				type(ndev) == "string" and ndev or "unknown"))
			io.stderr:flush()
			return false
		end
	end
	g_netdev = ndev

	local ttydev = fetch_ttyusb(g_4ginfo[g_usbid]["ttyindex"])
	if ttydev then
		-- write /tmp/4g_modem_tty with the control path of ttyUSB device
		local fhdl = io.open(TTY_MODEM, "wb")
		if fhdl then
			fhdl:write(gfmt("%s\n", ttydev))
			fhdl:close(); fhdl = nil
		end
	end
	return true
end

local function network_okay(iface)
	local cmds = { "ping", "-c3", "-w12" }
	local cmdidx = 0x1 + #cmds
	if type(iface) == "string" then
		cmds[cmdidx] = "-I"
		cmds[cmdidx + 0x1] = iface
		cmdidx = cmdidx + 0x2
	end
	for _, hname in ipairs(PING_ADDRS) do
		cmds[cmdidx] = hname
		local okay, output = invoker.invoke(invoker.OUTPUT + invoker.NOSTDIO, cmds)
		if type(output) == "string" and
			string.find(output, 'packet loss', 1, true) and
			not string.find(output, '100% packet loss', 1, true) then
			return hname
		end
		if okay == 0 then return hname end
		if invoker.tcpcheck(hname, 3883, 1500) then return hname end
		-- if invoker.tcpcheck(hname, 1883, 1500) then return hname end
		-- if invoker.tcpcheck(hname, 80, 1500) then return hname end
		-- if invoker.tcpcheck(hname, 443, 1500) then return hname end
	end
	return false
end

local function net4g_update_metric(iface, metric)
	local okay, output = invoker.invoke(invoker.OUTPUT,
		"nmcli", "-m", "tabular", "-t", "con", "show")
	if okay ~= 0 or type(output) ~= "string" then output = "" end

	local ndels = 0
	for line in string.gmatch(output, "[^\r\n]+") do
		local con = string.match(line, "^([^:]+)")
		if con and #con > 0 then
			local cinfo = nil
			okay, cinfo = invoker.invoke(invoker.OUTPUT + 0x70000,
				"nmcli", "con", "show", con)
			if okay == 0 and type(cinfo) == "string" and string.find(cinfo, iface, 1, true) then
				ndels = ndels + 1
				invoker.invoke(invoker.NOSTDIO, "nmcli", "con", "down", con)
				invoker.invoke(invoker.NOSTDIO, "nmcli", "con", "delete", con)
				io.stderr:write(gfmt("Delete NetworkManager connection: %s\n", con))
				io.stderr:flush()
			end
		end
	end

	invoker.invoke(invoker.NOSTDIO, "ip", "addr", "flush", "dev", iface)
	if ndels > 0 then invoker.waitsec(2) end
	local conn = gfmt('net4g_%s', iface)
	okay = invoker.invoke(0, "nmcli", "con", "add", "con-name",
		conn, "ifname", iface, "type", "ethernet",
		"ipv4.method", 'auto', "autoconnect", "yes",
		"ipv4.route-metric", tostring(metric))
	if okay ~= 0 then
		io.stderr:write("Error, failed to add new connection for 4G network\n")
		io.stderr:flush()
	end
	invoker.invoke(0, "nothup", "nmcli", "con", "up", conn)
	return true
end

local function network_connect(iface, metric)
	-- kill dhclient application first
	local i4g = g_4ginfo[g_usbid]

	local dftdev = gfmt('/dev/ttyUSB%d', i4g.ttyindex)
	update_sim_iccid(dftdev)
	-- check for /dev/ttyUSBx device
	local devn = fetch_ttyusb(i4g.ttyindex) or dftdev
	local devi = posix.stat(devn)
	if type(devi) ~= "table" or devi["type"] ~= "character device" then
		io.stderr:write(gfmt("Error, ttyUSB device not found: %s\n", devn))
		io.stderr:flush()
		return false
	end

	-- invoke comgt to establish network connection
	local okay = invoker.invoke(0, 'comgt', '-d', devn, '-s',
		gfmt('/opt/lnxall/etc/gcom/%s', i4g.script))
	if okay ~= 0 then
		io.stderr:write(gfmt("Error, comgt has returned %d\n",
			type(okay) == "number" and okay or 1))
		io.stderr:flush()
		-- return false -- contionue anyway
	end

	local count = 0
	invoker.msleep(3000) -- delay 3 seconds before continue
	-- wait for `iface network to show up
	while true do
		count = count + 1
		if count >= 10 then break end
		if netdev_exists(iface) then break end
		invoker.msleep(1000)
	end

	invoker.msleep(1000) -- again, delay one second
	-- check again for the existence of `iface
	if not netdev_exists(iface) then
		io.stderr:write(gfmt("Error, network device not found: %s\n", iface))
		io.stderr:flush()
		return false
	end

	if i4g.dhcp then
		net4g_update_metric(iface, metric)
		invoker.msleep(3000) -- delay 3 seconds for dhclient to acquire ipv4 address
	end
	return true
end

local function watch_resolv(delay)
	local flags = bit32.bor(invoker.IN_CREATE, invoker.IN_DELETE_SELF,
		invoker.IN_MODIFY, invoker.IN_CLOSE_WRITE)
	local rfile = '/etc/resolv.conf'
	if not posix.access(rfile) then check_resolv() end

	local changed, errmsg = invoker.watchfile(flags, delay * 1000, nil, rfile)
	if changed == nil then
		if type(errmsg) ~= "string" then errmsg = "unknown watchfile error" end
		io.stderr:write(errmsg); io.stderr:write("\n"); io.stderr:flush()
		invoker.waitsec(delay)
		return false
	end
	if changed and changed > 0 then
		invoker.msleep(1500) -- delay 1.5 second to avoid modifying by too many processes
		check_resolv()
	end
	return true
end

local function isat_time(hour)
	local nowt = os.date("*t")
	return nowt["hour"] == hour
end

local function mainloop()
	local midx, dcnt = 0, 0
	local nowtim = invoker.uptime()
	local nextim = nowtim + CHECK_INTERVAL
	update_sim_iccid()
	while true do
		check_resolv() -- check resolv.conf before continue
		local phost = network_okay()
		if phost then
			io.stdout:write(gfmt("%s: Network ping okay: %s\n", os.date(), phost))
			io.stdout:flush()
			if midx > 0 then midx = 0 end
			if dcnt > 0 then dcnt = 0 end
		else
			dcnt = dcnt + 1 -- increment disconnected
			if LAN_NETDEV then
				local rval = power_offon_4g(true)
				pci_usb_rebind(false)
				if rval then check_resolv(); watch_resolv(8) end
			else pci_usb_rebind(false); invoker.waitsec(5) end

			if isat_time(23) and dcnt >= 4 then -- it is 23 at midnight, safe to reboot
				write_logfile(LOG_FILE, gfmt("%s: reboot due to 4G network failure: %d.\n", os.date(), dcnt))
				invoker.invoke(invoker.NOSTDIO, "reboot"); invoker.waitsec(5)
			end

			pci_usb_rebind(true); invoker.waitsec(8)
			if g_netdev and not netdev_exists(g_netdev) then
				write_logfile(LOG_FILE, gfmt("%s: Error, no 4G net device found: %d\n", os.date(), dcnt))
			else
				midx = midx + 1 -- increment metric index value
				if midx > #g_metrics then midx = 1 end
				invoker.invoke(invoker.NOSTDIO, "ip", "addr", "flush", g_netdev)
				write_logfile(LOG_FILE, gfmt("%s: reconnect 4G network, count: %d\n", os.date(), dcnt))
				network_connect(g_netdev, g_metrics[midx])
			end
		end

		check_resolv() -- check resolv.conf again
		nowtim = invoker.uptime()
		-- delay until next checking time
		while nowtim < nextim do
			watch_resolv(nextim - nowtim)
			nowtim = invoker.uptime()
		end
		-- increment next checking time
		nextim = nextim + CHECK_INTERVAL
		if nextim <= nowtim then nextim = nextim + CHECK_INTERVAL end
	end
end

local initerr = 0
replace_grub('/boot/grub2/grub.cfg')
replace_grub('/boot/efi/EFI/centos/grub.cfg')
while not n4g_init(arg[1], arg[2], arg[3], arg[4]) do
	initerr = initerr + 1
	if (initerr % 4) == 0 and not network_okay() then
		if isat_time(23) then
			write_logfile(LOG_FILE, gfmt("%s: reboot due to 4G initialization failure.\n", os.date()))
			invoker.invoke(invoker.NOSTDIO, 'reboot'); invoker.waitsec(3)
		end
		pci_usb_rebind(false)
		if LAN_NETDEV then power_offon_4g() end
		invoker.waitsec(5); pci_usb_rebind(true); invoker.waitsec(2)
	end
	invoker.waitsec(60)
end
io.stdout:write("4G modem found, entering loop...\n")
io.stdout:flush()
mainloop() -- application hangs in the function
os.exit(2)
