#!/opt/lnxall/bin/lua

-- Created by jiaqiang.ye@lnxall.com
-- Simple LAN/WAN detection helper script for Combo/CentOS-7
-- 2023/09/10

-- load Lua modules
local posix = require 'posix'
local invoker = require 'invoker'

-- global variable definition
local SIG_CNT = 0
local TMP_RULES = "/tmp/60-net.rules"
local NAMES = { 'enp3s0', 'enp1s0', 'enp2s0' }

local function netdev_exist(netdev)
	local eval, output = invoker.invoke(invoker.NOSTDIO + invoker.OUTPUT,
		"ip", "link", "show", "dev", netdev)
	if eval == 0 and type(output) == "string" then
		return true
	end
	return false
end

-- function to enumerate PCI attached ethernet ports
local function enum_pci_eths(expect)
	local eval, output = invoker.invoke(invoker.OUTPUT, "lspci")
	if eval ~= 0 or type(output) ~= "string" then
		io.stderr:write("Error, invalid output of `lspci\n")
		io.stderr:flush()
		return nil
	end

	local ports = {}
	if not expect then expect = 0x3 end
	for pci in string.gmatch(output, "(%d%d:%d%d%.%d)%s+Ethernet controller") do
		local busnum = string.format('0000:%s', pci)
		if posix.access('/sys/bus/pci/devices/' .. busnum) ~= 0 then
			io.stderr:write(string.format("Error, PCI device not found: %s\n", busnum))
			io.stderr:flush()
		else
			local netdev = posix.glob(string.format('/sys/bus/pci/devices/%s/net/*', busnum))
			if type(netdev) == "table" and type(netdev[1]) == "string" then
				local ndev = string.match(netdev[1], "/([^/]+)$")
				if ndev and netdev_exist(ndev) then
					ports[#ports + 1] = { ["old"] = ndev, ["bus"] = busnum }
					io.stdout:write(string.format("Found PCI ethernet controller: %s => %s\n", busnum, ndev))
					io.stdout:flush()
				else
					io.stderr:write(string.format("Error, invalid network device: %s\n", ndev and ndev or 'wth'))
					io.stderr:flush()
				end
			else
				io.stderr:write("Error, no network device found for PCI port: %s\n", busnum)
				io.stderr:flush()
			end
		end
	end

	-- check the number of ethernet controllers found:
	if expect ~= #ports then
		io.stderr:write(string.format("Error, invalid number of PCI ethernet port found: %d\n", #ports))
		io.stderr:flush()
		return nil
	end
	return ports
end

local function link_detected(netdev)
	local eval, output = invoker.invoke(invoker.OUTPUT, "ethtool", netdev)
	if eval ~= 0 or type(output) ~= "string" then
		io.stderr:write(string.format("Error, invalid output from `ethtool %s\n", netdev))
		io.stderr:flush()
		return false
	end

	local link = string.match(output, "Link detected:%s+(%w+)")
	if not link then
		io.stderr:write(string.format("Error, invalid link status for %s\n", netdev))
		io.stderr:flush()
		return false
	end

	return link == "yes"
end

local function check_ethlinks(enets, newdev)
	local linked = 0
	for _, ethi in ipairs(enets) do
		local link = link_detected(ethi["old"])
		if link then
			linked = linked + 1
			if newdev and not ethi["new"] then
				ethi["new"] = newdev
				newdev = nil
			end
		end
	end
	return linked
end

local function emit_msgwait(msg)
	if type(msg) ~= "string" then
		msg = "Press 'Enter' key to continue...\n"
	end
	io.stdout:write(msg)
	io.stdout:flush()
	invoker.pause()
end

local function dsighandler(signo)
	if signo ~= posix.SIGINT then
		io.stderr:write(string.format("Unexpected signal: %d\n", signo))
		io.stderr:flush()
		os.exit(1)
	end
	SIG_CNT = SIG_CNT + 0x1
	if SIG_CNT >= 0x3 then os.exit(2) end
	return true
end

local function mainfunc()
	posix.signal(posix.SIGINT, posix.SIG_DFL)
	posix.signal(posix.SIGINT, dsighandler)
	posix.setenv('PATH', '/opt/lnxall/bin:/usr/local/bin:/usr/bin:/usr/sbin:/bin:/sbin', 1)
	local eths = enum_pci_eths(0x3)
	if not eths then return false end

	----------------------------------------------------
	io.stdout:write("Detecting LAN network device...\n")
	io.stdout:flush()

	if check_ethlinks(eths, NAMES[1]) ~= 0x1 then
		io.stderr:write("Error, Retry with only LAN cable attached.\n")
		io.stderr:flush()
		return false
	end

	----------------------------------------------------
	emit_msgwait("LAN network device found.\nPlugin WAN1 cable and then press 'CTRL-C' to continue...\n")
	if check_ethlinks(eths, NAMES[2]) ~= 0x2 then
		io.stderr:write("Error, cannot determine WAN1 network device.\n")
		io.stderr:flush()
		return false
	end

	local rules = io.open(TMP_RULES, "wb")
	if not rules then
		io.stderr:write("Error, failed to open temporary rules file.\n")
		io.stderr:flush()
		return false
	end

	----------------------------------------------------
	for _, ethi in ipairs(eths) do
		if not ethi["new"] then
			ethi["new"] = NAMES[3]
		end

		io.stdout:write(string.format("Network device will be renamed: %s => %s\n",
			ethi["old"], ethi["new"]))
		rules:write(string.format('ACTION=="add", SUBSYSTEM=="net", SUBSYSTEMS=="pci", DRIVERS=="?*", ATTR{type}=="1", KERNELS=="%s", NAME="%s"\n',
			ethi["bus"], ethi["new"]))
	end
	rules:write('ACTION=="add", SUBSYSTEM=="net", DRIVERS=="?*", ATTR{type}=="1", PROGRAM="/lib/udev/rename_device", RESULT=="?*", NAME="$result"\n')
	rules:close(); rules = nil; posix.sync()
	if invoker.invoke(0, "mv", "-v", "-f", TMP_RULES, "/usr/lib/udev/rules.d/60-net.rules") ~= 0 then
		io.stderr("Error, failed to move net rules file.\n")
		io.stderr:flush()
		return false
	end
	posix.sync()

	----------------------------------------------------
	local ifcfg = '/etc/sysconfig/network-scripts/ifcfg-'
	for _, ethi in ipairs(eths) do
		local oldcfg = ifcfg .. ethi["old"]
		if ethi["old"] ~= ethi["new"] and posix.access(oldcfg) == 0 then
			io.stdout:write(string.format("Try to rename 'ifcfg-%s' as 'ifcfg-%s'...\n",
				ethi["old"], ethi["new"]))
			io.stdout:flush()
			invoker.invoke(0, "mv", "-v", "-f", oldcfg, ifcfg .. ethi["new"])
		end
	end

	io.stdout:write("Network devices renamed, please reboot your system\n")
	io.stdout:flush()
	return true
end

if mainfunc() then os.exit(0) end
os.exit(1)
